function [] = Jplot(Fname, Psave)
%Vplot makes a plot of existing data in the 'Fname' file.
%
%   Fname:    String of .mat file name - e.g. 'SUM_Tesla_W7_VSP.mat'
%   Psave:    If Psave=1 the plot is saved to file


%% LOAD DATA
% Get data: Size, T_CPU, T_GPU, TimeStamp, FCT, GPU, OS, ATP, PRC, L_CPU, L_GPU
%   Size:     Array with data for matrix/vector values
%   T_CPU:    Vector with execution times for CPU (in seconds)
%   T_GPU:    Vector with execution times for GPU (in seconds)
%   FCT:      Function to be benchmarked
%   GPU:      GPU type (9600MGT / Tesla / FX3800 / 9400M)
%   OS:       Operating system (OSX / Ubuntu / W7 / VST)
%   ATP:      Array type (Matrix/Vector)
%   PRC:      Precision (Single Precision / Double Precision)
%   L_CPU:    Legend for CPU type
%   L_GPU:    Legend for GPU type
Name = ['load ' Fname];
eval(Name);


%% PLOT RESULTS
%==========================================================================
if ATP=='V',   % If vector plot
    % Set up title
    ATPL = 'Vector';

    if PRC=='SP',
        PRCL = 'Single Precision (SP)';
    else
        PRCL = 'Double Precision (DP)';
    end
    F_TTL = [FCT ' - ' PRCL ', ' ATPL];

    % Make figure
    figure(1); clf(1);
    subplot(2,1,1);
    plot(Size/1E6, T_CPU*1E3, 'c-*', ...
        Size/1E6, T_GPU*1E3, 'm-x', 'LineWidth', 2);
    grid;
    title(F_TTL);
    xlabel('Vector size   [M-]');
    ylabel('Execution time   [ms]');
    legend(L_CPU, L_GPU, 'Location', 'NorthWest');
 
    subplot(2,1,2);
    plot(Size/1E6, T_CPU ./ T_GPU, 'r-*', 'LineWidth', 2);
    grid;
    xlabel('Vector size   [M-]');
    ylabel('Speed-up   [-]');
else          % If matrix plot
    % Set up title
    ATPL = 'Square Matrix';
    
    if PRC=='SP',
        PRCL = 'Single Precision (SP)';
    else
        PRCL = 'Double Precision (DP)';
    end
    F_TTL = [FCT ' - ' PRCL ', ' ATPL];

    % Make figure
    figure(1); clf(1);
    subplot(2,1,1);
    plot(Size, T_CPU*1E3, 'c-*', ...
        Size, T_GPU*1E3, 'm-x', 'LineWidth', 2);
    grid;
    title(F_TTL);
    xlabel('#Rows = #Columns   [-]');
    ylabel('Execution time   [ms]');
    legend(L_CPU, L_GPU, 'Location', 'NorthWest');
 
    subplot(2,1,2);
    plot(Size, T_CPU ./ T_GPU, 'r-*', 'LineWidth', 2);
    grid;
    xlabel('#Rows = #Columns   [-]');
    ylabel('Speed-up   [-]');
end


%% SAVE FIGURES IF REQUESTED
if Psave==1,
    jpgFname = [Name '.jpg'];
    print( gcf, '-djpeg99', '-r100', jpgFname );
    epsfFname = [Name '.eps'];
    print( gcf, '-depsc2', '-r200', epsfFname );
end

end