%%----------------------------------------------------------------

%   Product:   Jacket
%   Example:   black_scholes_example
%   Version:   1.2

%   Copyright (c) AccelerEyes LLC. All rights reserved.
%   See http://www.accelereyes.com/doc/EULA.pdf for details.

%   This software is distributed WITHOUT ANY WARRANTY; without even 
%   the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
%   PURPOSE.  See the above copyright notices for more information.

%%----------------------------------------------------------------
function black_scholes_example(mode)

  fprintf( '** Jacket Black-Scholes Example **\n' );
  fprintf( '**        by AccelerEyes        **\n\n' );

  % Set a default mode
  use_double = 0;
  a = gpu_entry(13);
  if a.compute == '1.3' 
    use_double = 1;
  end

  % Comply to mode, if given
  if nargin == 1
    if strcmp(mode,'single') == 1
      use_double = 0;
    elseif strcmp(mode, 'double') == 1
      use_double = 1;
    else
      disp('Incorrect mode. Reverting to default precision');
    end
  end

  % Use cache
  S = mfilename('fullpath'); S = S(1:end-length(mfilename));
  gcache('load',[S 'cache.jkt']);

  which cache.jkt
  % Load input data
  load('optionsData.mat');

  t1 = zeros(10,1);
  t2 = t1; t3 = t1;
  iter = 5;

  % Loop over different input sizes
  for N = 1:5:50,

    % Make copies of input data to make larger data set
    if use_double
      S = double(repmat(C{1},N,1));
      X = double(repmat(C{2},N,1));
      R = double(repmat(C{3},N,1));
      V = double(repmat(C{5},N,1));
      T = double(repmat(C{6},N,1));
    else      
      S = single(repmat(C{1},N,1));
      X = single(repmat(C{2},N,1));
      R = single(repmat(C{3},N,1));
      V = single(repmat(C{5},N,1));
      T = single(repmat(C{6},N,1));
    end
    % Create GPU copies of the data
    %% 'gforce' used to force thread synchrony in timing
    if use_double
      Sg = gdouble(S); gforce( Sg );
      Xg = gdouble(X); gforce( Xg );
      Rg = gdouble(R); gforce( Rg );
      Vg = gdouble(V); gforce( Vg );
      Tg = gdouble(T); gforce( Tg );
    else
      Sg = gsingle(S); gforce( Sg );
      Xg = gsingle(X); gforce( Xg );
      Rg = gsingle(R); gforce( Rg );
      Vg = gsingle(V); gforce( Vg );
      Tg = gsingle(T); gforce( Tg );
    end

    fprintf( ' Input Data size = %g x %g',size(S,1),size(S,2));
    
    % Compute on the CPU
    tic
    for i = 1:iter
      [Cc,Pc] = black_scholes(S,X,R,V,T);
    end
    t1(N) = toc;
    fprintf('\n Mean CPU Time = %d', t1(N));

    % Compute on the GPU
    tic
    for i = 1:iter
      [Cg,Pg] = black_scholes(Sg,Xg,Rg,Vg,Tg); 
      gforce(Cg, Pg);
    end
    t2(N) = toc;
    fprintf('\n Mean GPU Time = %d', t2(N));
    fprintf('\n Speedup (CPU time / GPU time) = %g\n\n',t1(N)./t2(N));
    
  end

  % Plot results
  fprintf( ' Plotting the Results.\n\n' );
  figure; subplot(1,2,1); hold on;
  plot( 4000*[1:5:50], t1(t1>0), 'r' );
  plot( 4000*[1:5:50], t2(t2>0), 'g' );
  hold off;
  title('Red=CPU, Green=GPU');
  ylabel('Runtime (ms)'); xlabel('Vector Length');
  subplot(1,2,2); hold on;
  plot( 4000*[1:5:50], t1(t1>0)./t2(t2>0), 'b' );
  title('Blue=GPU Speedup');
  ylabel('GPU Speedup'); xlabel('Vector Length');
  hold off;
  fprintf( ' DONE\n' );

end
